#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/vt.h>
#include <string.h>
#include <ctype.h>
#include <vga.h>
#include <vgagl.h>
#include <vgakeyboard.h>
#include "gpulocal.h"
#include "keyboard.h"

#define MODE	G640x480x64K
#define max_width	640
#define max_height	480

static int cur_width,cur_height;
static UINT16 scr[max_width*max_height];
static int keybev;
static UINT16 keybkey;
static UINT16 dsp_keymap[km_s];

static int dsp_init();
static void dsp_exit();
static void dsp_open_window(unsigned int w,unsigned int h);
static void dsp_resize(unsigned int w,unsigned int h);
static void dsp_change_title(char *title);
static void dsp_render_frame(UINT32 *image,int start,int endw);
static int dsp_key_pressed();
static UINT16 dsp_get_key();

static int dsp_check();
static void dsp__keybupdate(int key, int press);
static void dsp_fillkeymap();


DisplayStruct Dsp_VGA = {
	dsp_init,
	dsp_exit,
	dsp_open_window,
	dsp_resize,
	dsp_change_title,
	dsp_render_frame,
	dsp_key_pressed,
	dsp_get_key
};

static int dsp_init() {
	int white;
	if (dsp_check()) return -1;
	vga_disabledriverreport();
	vga_init(); // should be always successful

	if (!vga_hasmode(MODE))
	    return -1;

	vga_setmode(MODE);
	gl_setcontextvga(MODE);

	white = vga_white();

	gl_setfont(8,8,gl_font8x8);
	gl_setwritemode(FONT_COMPRESSED + WRITEMODE_OVERWRITE);
	gl_setfontcolors(0,white);
	
	if (keyboard_init()==-1) return -1;
	keyboard_seteventhandler(dsp__keybupdate);

	dsp_fillkeymap();
	return 0;
}

static int dsp_check() {
	int console;

	console = STDIN_FILENO;
	if (console>=0) {
	    struct stat sb;
	    struct vt_mode vt;
	    if ((fstat(console, &sb) < 0) | (ioctl(console, VT_GETMODE, &vt) < 0))
	    console = -1;
	    }

	return (!(geteuid() == 0) & (console >= 0));
}


static void dsp_open_window(unsigned int w,unsigned int h) {
	int x,y;

	gl_clearscreen(0);
	x = (max_width-w)/2;
	y = (max_height-h)/2;
	gl_setclippingwindow(x,y,w+x-1,h+y-1);
	cur_width=w;
	cur_height=h;
}

static void dsp_resize(unsigned int w,unsigned int h) {
	dsp_open_window(w,h);
}

static void dsp_exit() {
	keyboard_close();
	vga_setmode(TEXT);
}

static void dsp_change_title(char *title) {
/*	int x,y;

	gl_setclippingwindow(0,0,max_width,max_height);
	gl_printf (5,5,title);
	x = (max_width-cur_width)/2;
	y = (max_height-cur_height)/2;
	gl_setclippingwindow(x,y,cur_width+x-1,cur_height+y-1);*/
}

static void dsp_render_frame(UINT32 *image,int start,int endw) {
	int i,j,k=0;
	UINT16 *vr;

	vr=(UINT16*)(image+start);
	for (j=0;j<cur_height;j++) { // from dsp_x11.c
	    for (i=0;i<cur_width;i++) {
		int t=*(vr++);
		t = (t&RMASK)|((t&GMASK)<<1)|((t&BMASK)<<1);
		scr[k++]=t;
	    }
	    vr+=(endw-cur_width);
	    }
	gl_putbox((max_width-cur_width)/2,(max_height-cur_height)/2,cur_width,cur_height,scr);
}

static int dsp_key_pressed() {
	keybev = -1;
	if (keyboard_update()==0) return NO_KEY;

	while (keybev == -1) {}

	switch (keybev) {
	    case KEY_EVENTRELEASE: return RELEASED_KEY;
	    case KEY_EVENTPRESS: return PRESSED_KEY;
	    }
	return NO_KEY;
}

static void dsp__keybupdate(int key, int press) {
	keybev = press;
	keybkey = key;
}

static UINT16 dsp_get_key() {
	int i;

	for (i=0; i<km_s; i++) if (keybkey == dsp_keymap[i]) return i;
	return KEY_INVALID;
}


static void dsp_fillkeymap() {
	int i;

	memset(dsp_keymap,0,sizeof(dsp_keymap));

	dsp_keymap[KEY_0] = SCANCODE_0;
	for (i=1;i<10;i++) dsp_keymap[KEY_0+i] = SCANCODE_1+(i-1);

	dsp_keymap[KEY_UP] = SCANCODE_CURSORUP;
	dsp_keymap[KEY_DOWN] = SCANCODE_CURSORDOWN;
	dsp_keymap[KEY_LEFT] = SCANCODE_CURSORLEFT;
	dsp_keymap[KEY_RIGHT] = SCANCODE_CURSORRIGHT;

	dsp_keymap[KEY_ESCAPE] = SCANCODE_ESCAPE;
	dsp_keymap[KEY_RETURN] = SCANCODE_ENTER;
	dsp_keymap[KEY_SPACE] = SCANCODE_SPACE;
	dsp_keymap[KEY_a] = SCANCODE_A;
	dsp_keymap[KEY_b] = SCANCODE_B;
	dsp_keymap[KEY_c] = SCANCODE_C;
	dsp_keymap[KEY_d] = SCANCODE_D;
	dsp_keymap[KEY_e] = SCANCODE_E;
	dsp_keymap[KEY_f] = SCANCODE_F;
	dsp_keymap[KEY_g] = SCANCODE_G;
	dsp_keymap[KEY_h] = SCANCODE_H;
	dsp_keymap[KEY_i] = SCANCODE_I;
	dsp_keymap[KEY_j] = SCANCODE_J;
	dsp_keymap[KEY_k] = SCANCODE_K;
	dsp_keymap[KEY_l] = SCANCODE_L;
	dsp_keymap[KEY_m] = SCANCODE_M;
	dsp_keymap[KEY_n] = SCANCODE_O;
	dsp_keymap[KEY_o] = SCANCODE_P;
	dsp_keymap[KEY_p] = SCANCODE_Q;
	dsp_keymap[KEY_q] = SCANCODE_R;
	dsp_keymap[KEY_s] = SCANCODE_S;
	dsp_keymap[KEY_t] = SCANCODE_T;
	dsp_keymap[KEY_u] = SCANCODE_U;
	dsp_keymap[KEY_v] = SCANCODE_V;
	dsp_keymap[KEY_w] = SCANCODE_W;
	dsp_keymap[KEY_x] = SCANCODE_X;
	dsp_keymap[KEY_y] = SCANCODE_Y;
	dsp_keymap[KEY_z] = SCANCODE_Z;

	for (i=0;i<10;i++) dsp_keymap[KEY_F1+i] = SCANCODE_F1+i;
	dsp_keymap[KEY_F11] = SCANCODE_F11;
	dsp_keymap[KEY_F12] = SCANCODE_F12;

	// finish this
}
